﻿package com.kraftner.justlineon.lineservers 
{
	import com.gskinner.utils.Rndm;
	import com.kraftner.communication.ControllerConnection;
	import com.kraftner.justlineon.AbortRule;
	import JustLineOn;
	import com.kraftner.justlineon.Segment;
	import flash.events.Event;
	import flash.events.KeyboardEvent;
	import flash.geom.Point;
	
	/**
	 * @internal
	 * CircularImplosion.as
	 * Copyright (c) 2008 Thomas Kräftner
	 * 
	 * Visit http://blog.kraftner.com for documentation, updates and more free code.
	 */
	
	/**
	* This LineServer always starts from the boundaries of the Stage and moves until it hits the end of the stage or a certain distance from the center. Each new line starts some degree rotated relative to the former line. The starting distance from the center is constantly decreased/increased so a spiral of lines is formed.
	* @author Thomas Kräftner
	* @see com.kraftner.justlineon.lineservers.CircularImplosion
	*/
	public class CircularImplosion extends LineServer
	{
		private var center:Point = JustLineOn.center;
		private var angle:Number = 0;
		private var away:Number;
		
		private var length:Number;
		private var angleMin:Number;
		private var angleMax:Number;
		private var awayStart:Number;
		private var awayEnd:Number;
		private var awaySub:Number;
		private var breakInChance:Number;
		private var randomAbort:Number
		private var circleStepWidth:Number;
		private var restart:Boolean;
		
		/**
		 * @param	length The average length of each line Segment. The actual length of a Segment varies between 60% and 140% of this value.
		 * @param	angleDegreeMin Minimum angle the new Segment turns away from the direction of the previous Segment
		 * @param	angleDegreeMax Maximum angle the new Segment turns away from the direction of the previous Segment
		 * @param	awayStart Distance at which a new line starts. This only applies in connection with the awaySub parameter.
		 * @param	awayEnd Minimum distance the line can be away from the center. This only applies in connection with the breakInChance parameter.
		 * @param	awaySub Factor which is substracted from awayStart on each new line. Can be negative as well.
		 * @param	breakInChance Chance the line ignores the away parameter and moves on. This Chance is represented by a Number between 0 and 1, 0 beeing impossible to break in. This chance isn't a one time decision but is triggered on each Segment of the line again.
		 * @param	randomAbort Chance the line randomly aborts. This Chance is represented by a Number between 0 and 1, 0 meaning the line never randomly aborts and 1 meaning it always aborts. If you set this parameter to 1 the line won't draw at all.
		 * @param	circleStepWidthDegree Number of degrees by which the next line starts offset from the previous angle.
		 * @param	restart true means the Spiral restarts when awayStart is too close to awayEnd to draw a Segment
		 * @see com.kraftner.justlineon.AbortRule#CloseToCenter()
		 * @see com.kraftner.justlineon.AbortRule#AwayFromCenter()
		 */
		public function CircularImplosion(length:Number,angleDegreeMin:Number, angleDegreeMax:Number, awayStart:Number, awayEnd:Number, awaySub:Number, breakInChance:Number, randomAbort:Number, circleStepWidthDegree:Number,restart:Boolean):void
		{
			this.length = length;
			this.angleMin = (Math.PI / 180) * angleDegreeMin;
			this.angleMax = (Math.PI / 180) * angleDegreeMax;
			this.awayStart =awayStart;
			this.awayEnd = awayEnd;
			this.awaySub = awaySub;
			this.breakInChance = 1-breakInChance;
			this.randomAbort = randomAbort;
			this.circleStepWidth = (Math.PI / 180) * circleStepWidthDegree;
			this.restart = restart;
			
			away = awayStart;
			
			var startVector:Point = new Point(Rndm.float( -1, 1), Rndm.float( -1, 1));
			startVector.normalize(awayStart);
			var startPoint:Point = center.add(startVector);
			startVector.normalize( -length);
				
			prev_Segment = new Segment(startPoint.x + startVector.x, startPoint.y + startVector.y, new Segment(startPoint.x, startPoint.y), true);
			angle = prev_Segment.angle;
			
			var controllerConnection:ControllerConnection = ControllerConnection.getInstance();
			controllerConnection.addRange("length", setLength, 20, 150, length);
			controllerConnection.addRange("angleDegreeMin", setAngleDegreeMin, -360, 360, angleDegreeMin);
			controllerConnection.addRange("angleDegreeMax", setAngleDegreeMax, -360, 360, angleDegreeMax);
			controllerConnection.addRange("awayStart", setAwayStart, 1, 500, awayStart);
			controllerConnection.addRange("awayEnd", setAwayEnd, 1, 500, awayEnd);
			controllerConnection.addRange("awaySub", setAwaySub, 0, 10, awaySub);
			controllerConnection.addRange("breakInChance", setBreakInChance, 0, 1, breakInChance);
			controllerConnection.addRange("randomAbort", setRandomAbort, 0, 1, randomAbort);
			controllerConnection.addRange("circleStepWidthDegree", setCircleStepWidthDegree, -360, 360, circleStepWidthDegree);
			controllerConnection.addChoice("restart", setRestart, 2, uint(restart));
		}
		
		private function setLength(i:Number):void { length=i; }
		private function setAngleDegreeMin(i:Number):void { angleMin = (Math.PI / 180) * i; }
		private function setAngleDegreeMax(i:Number):void { angleMax = (Math.PI / 180) * i; }
		private function setAwayStart(i:Number):void { awayStart = i; }
		private function setAwayEnd(i:Number):void { awayEnd = i; }
		private function setAwaySub(i:Number):void { awaySub = i; }
		private function setBreakInChance(i:Number):void { breakInChance = 1-i; }
		private function setRandomAbort(i:Number):void { randomAbort = i; }
		private function setCircleStepWidthDegree(i:Number):void { circleStepWidth = (Math.PI / 180) * i; }
		private function setRestart(i:Number):void { restart = Boolean(i); }
		
		/**
		 * @inheritDoc
		 */
		override protected function get newSegment():Segment
		{
			var newSegment:Segment = new Segment(1, 1,prev_Segment);
			newSegment.angle = angle+Rndm.float(angleMin,angleMax);
			newSegment.length = length * Rndm.float(0.6, 1.4);
			
			if (prev_Segment.start)
			{
				angle += circleStepWidth;
				
				var tempSegment:Segment = new Segment(1, 1, new Segment(center.x, center.y));
				tempSegment.angle = angle;
				tempSegment.length = -away;
				
				var startVector:Point = new Point(tempSegment.rel_x, tempSegment.rel_y);
				startVector.normalize(-length);
				
				newSegment = new Segment(tempSegment.x + startVector.x, tempSegment.y + startVector.y, new Segment(tempSegment.x, tempSegment.y,null,true));
			}
			
			if (Rndm.boolean(breakInChance) && AbortRule.CloseToCenter(new Point(newSegment.x,newSegment.y),awayEnd) || Rndm.boolean(randomAbort) && !prev_Segment.start || AbortRule.AwayFromCenter(new Point(newSegment.x, newSegment.y), awayStart) )
			{
				newSegment.start = true;
			}
			
			if (away > awayEnd + length + awaySub) away -= awaySub;
			else { if (restart) { away = awayStart; } }
			
			return newSegment;
		}
		
	}
	
}