﻿package com.kraftner.justlineon.lineservers 
{
	import com.gskinner.utils.Rndm;
	import com.kraftner.justlineon.AbortRule;
	import com.kraftner.justlineon.Segment;
	import flash.geom.Point;
	import JustLineOn;
	import com.kraftner.communication.ControllerConnection;
	
	/**
	 * @internal
	 * AlwaysTurn.as
	 * Copyright (c) 2008 Thomas Kräftner
	 * 
	 * Visit http://blog.kraftner.com for documentation, updates and more free code.
	 */
	
	/**
	* This LineServer always starts from the center and moves until it hits the end of the stage or a certain distance from the center.
	* @author Thomas Kräftner
	* @see com.kraftner.justlineon.lineservers.AlwaysTurnImplosion
	*/
	public class AlwaysTurn extends LineServer
	{
		private var center:Point = JustLineOn.center;
		private var length:Number;
		private var lengthMulti:Number;
		private var lengthAdd:Number;
		private var angleMin:Number;
		private var angleMax:Number;
		private var away:Number;
		private var breakOutChance:Number;
		private var randomAbort:Number
		
		/**
		 * @param	length The length each line starts with. This is continually increased/decreased according to the following two parameters.
		 * @param	lengthMulti Factor by which the length of the previous Segment is multiplied to get the length of the new Segment.
		 * @param	lengthAdd Factor which is added to the length of the previous Segment to get the length of the new Segment. Can be negative as well.
		 * @param	angleDegreeMin Minimum angle the new Segment turns away from the direction of the previous Segment
		 * @param	angleDegreeMax Maximum angle the new Segment turns away from the direction of the previous Segment
		 * @param	away Maximum distance the line can be away from the center. This only applies in connection with the next parameter.
		 * @param	breakOutChance Chance the line ignores the away parameter and moves on. This Chance is represented by a Number between 0 and 1, 0 beeing impossible to break out. This chance isn't a one time decision but is triggered on each Segment of the line again.
		 * @param	randomAbort Chance the line randomly aborts. This Chance is represented by a Number between 0 and 1, 0 meaning the line never randomly aborts and 1 meaning it always aborts. If you set this parameter to 1 the line won't draw at all.
		 * @see com.kraftner.justlineon.AbortRule#AwayFromCenter()
		 * @see com.kraftner.justlineon.AbortRule#OutOfStage()
		 */
		public function AlwaysTurn(length:Number, lengthMulti:Number, lengthAdd:Number, angleDegreeMin:Number, angleDegreeMax:Number, away:Number, breakOutChance:Number, randomAbort:Number):void
		{
			this.length = length;
			this.lengthMulti = lengthMulti;
			this.lengthAdd = lengthAdd;
			this.angleMin = (Math.PI / 180) * angleDegreeMin;
			this.angleMax = (Math.PI / 180) * angleDegreeMax;
			this.away = away;
			this.breakOutChance = 1-breakOutChance;
			this.randomAbort = randomAbort;
			
			prev_Segment = new Segment(center.x + .1, center.y + .1, new Segment(center.x, center.y),true);
			prev_Segment.angle = Rndm.float(0, Math.PI * 2);
			prev_Segment.length = .1;
			
			var controllerConnection:ControllerConnection = ControllerConnection.getInstance();
			controllerConnection.addRange("length", setLength, 20, 150, length);
			controllerConnection.addRange("lengthMulti", setLengthMulti, 0.5, 1, lengthMulti);
			controllerConnection.addRange("lengthAdd", setLengthMulti, 0, 20, lengthAdd);
			controllerConnection.addRange("angleDegreeMin", setAngleDegreeMin, -360, 360, angleDegreeMin);
			controllerConnection.addRange("angleDegreeMax", setAngleDegreeMax, -360, 360, angleDegreeMax);
			controllerConnection.addRange("away", setAway, 1, 500, away);
			controllerConnection.addRange("breakOutChance", setBreakOutChance, 0, 1, breakOutChance);
			controllerConnection.addRange("randomAbort", setRandomAbort, 0, 1, randomAbort);
		}
		
		private function setLength(i:Number):void { length=i; }
		private function setLengthMulti(i:Number):void { lengthMulti=i; }
		private function setLengthAdd(i:Number):void { lengthAdd=i; }
		private function setAngleDegreeMin(i:Number):void { angleMin = (Math.PI / 180) * i; }
		private function setAngleDegreeMax(i:Number):void { angleMax = (Math.PI / 180) * i; }
		private function setAway(i:Number):void { away = i; }
		private function setBreakOutChance(i:Number):void { breakOutChance = 1-i; }
		private function setRandomAbort(i:Number):void { randomAbort = i; }
		
		/**
		 * @inheritDoc
		 */
		override protected function get newSegment():Segment
		{
			var newSegment:Segment = new Segment(1, 1, prev_Segment);
			
			newSegment.angle = prev_Segment.angle + Rndm.float(angleMin,angleMax);
			newSegment.length = prev_Segment.length * lengthMulti +lengthAdd;
			
			if (prev_Segment.start)
			{
				newSegment.angle = Rndm.float(0, Math.PI * 2);
				newSegment.length = length * Rndm.float(0.6, 1.4);;
			}
			if (newSegment.length < 1 || (Rndm.boolean(breakOutChance) && AbortRule.AwayFromCenter(new Point(newSegment.x,newSegment.y),away)) || Rndm.boolean(randomAbort) || AbortRule.OutOfStage(new Point(newSegment.x,newSegment.y)))
			{
				newSegment = new Segment(center.x + .1, center.y + .1, prev_Segment,true);
			}
			return newSegment;
		}
		
	}
	
}