﻿package com.kraftner.geom 
{
	import flash.utils.getQualifiedClassName;
	
	/**
	 * @internal
	 * Palette.as
	 * Copyright (c) 2008 Thomas Kräftner
	 * 
	 * Visit http://blog.kraftner.com for documentation, updates and more free code.
	 */
	
	/**
	* Template for all kinds of Palettes.
	* <p>This Class defines the base functionality of all Palettes which includes sorting methods and access to the contained colors.</p>
	* <p>This Class should not be instantiated directly. Use child classes like StaticPalette instead.</p>
	* @see com.kraftner.geom.StaticPalette
	* @author Thomas Kräftner
	*/
	public class Palette
	{
		/**
		 * Holds the Color Objects of this palette.
		 */
		protected var palette:Vector.<Color>;
		
		private var pointer:uint = 0;
		
		/**
		 * Constructor method. This Class should not be instantiated directly. Use child classes like StaticPalette instead.
		 * @see com.kraftner.geom.StaticPalette
		 */
		public function Palette()
		{
			if (getQualifiedClassName(this) == "com.kraftner.geom::Palette")	throw(new Error("Palette shouldn't be instantiated directly."));
			palette = new Vector.<Color>();
		}
		
		/**
		 * Number of Colors on this Palette
		 */
		public function get length():uint { return palette.length; }
		
		/**
		 * Returns the next Color of this Palette. This moves the pointer of this palette to the next Color.
		 * @return next Color Object
		 * @see #getIndex()
		 * @see #prev()
		 * @see #current()
		 * @see #extend()
		 */
		public function next():Color { return getIndex(pointer+1); }
		
		/**
		 * Returns the previous Color of this Palette. This moves the pointer of this palette to the previous Color.
		 * @return previous Color Object
		 * @see #getIndex()
		 * @see #next()
		 * @see #current()
		 * @see #extend()
		 */
		public function prev():Color { return getIndex(pointer-1); }
		
		/**
		 * Returns the current Color of this Palette.
		 * @return current Color Object
		 * @see #getIndex()
		 * @see #prev()
		 * @see #next()
		 * @see #extend()
		 */
		public function current():Color { return getIndex(pointer); }
		
		/**
		 * Returns the color at a certain position of this Palette.
		 * @param	index Index of the Color to get. <p>If the index is out of range and the extend function doesn't extend the Palette the first Color of the Palette is returned.</p>
		 * @param	movePointer Change the pointer to the Color that is returned.
		 * @return Color Object at the index-Position
		 * @see #prev()
		 * @see #next()
		 * @see #current()
		 * @see #extend()
		 */
		public function getIndex(index:uint, movePointer:Boolean = true):Color
		{
			extend(index);
			if (index >= length || index < 0)
			{
				//trace("restarting");
				return getIndex(0, movePointer);
			}else{
				if (movePointer) pointer = index;
				return palette[index];
			}
		}
		
		/**
		 * Defines the rule by which a Palette extends itself when the end is reached.
		 * @param	index
		 * @see #getIndex()
		 */
		protected function extend(index:uint):void { }
		
		/**
		 * Returns a Color Object representing the average color of this Palette
		 * @return average Color
		 */
		public function average():Color
		{
			var ar:Number=0;
			var ag:Number=0;
			var ab:Number=0;
			for (var i:int = 0; i < length-1; i++) 
			{
				var color:Color = palette[i];
				ar += color.r;
				ag += color.g;
				ab += color.b;
			}
			return new Color(ar/length,ag/length, ab / length);
		}
		
		//////////////////////////////Sorting Methods. Sort by all Properties of Color//////////////////////////////
		
		
		/**
		 * Sorts the Colors of this Palette by the red part of the contained Colors.
		 */	
		public function sortByRed():void { palette.sort(sortRed); }
		private function sortRed(c1:Color, c2:Color):Number
		{
			if (c1.r < c2.r) return -1;
			else if (c1.r > c2.r) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the green part of the contained Colors.
		 */
		public function sortByGreen():void	{ palette.sort(sortGreen);	}
		private function sortGreen(c1:Color, c2:Color):Number
		{
			if (c1.g < c2.g) return -1;
			else if (c1.g > c2.g) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the blue part of the contained Colors.
		 */
		public function sortByBlue():void	{ palette.sort(sortBlue);	}
		private function sortBlue(c1:Color, c2:Color):Number
		{
			if (c1.b < c2.b) return -1;
			else if (c1.b > c2.b) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the HEX value of the contained Colors. <p>The result of this might be unexpected and of little practical usefullness, but for matters of completeness this sorting Method is still offered.</p>
		 */
		public function sortByHEX():void	{ palette.sort(sortHEX);	}
		private function sortHEX(c1:Color, c2:Color):Number
		{
			if (c1.hex < c2.hex) return -1;
			else if (c1.hex > c2.hex) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the brightness of the contained Colors.
		 */
		public function sortByBrightness():void	{ palette.sort(sortBrightness);	}
		private function sortBrightness(c1:Color, c2:Color):Number
		{
			if (c1.brightness < c2.brightness) return -1;
			else if (c1.brightness > c2.brightness) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the saturation (as set by the HSV Colorsystem) of the contained Colors.
		 */
		public function sortBySaturationHSV():void	{ palette.sort(sortSaturationHSV);	}
		private function sortSaturationHSV(c1:Color, c2:Color):Number
		{
			if (c1.saturation_HSV < c2.saturation_HSV) return -1;
			else if (c1.saturation_HSV > c2.saturation_HSV) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the saturation (as set by the HSL Colorsystem) of the contained Colors.
		 */
		public function sortBySaturationHSL():void	{ palette.sort(sortSaturationHSL);	}
		private function sortSaturationHSL(c1:Color, c2:Color):Number
		{
			if (c1.saturation_HSL < c2.saturation_HSL) return -1;
			else if (c1.saturation_HSL > c2.saturation_HSL) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the lightness (as set by the HSL Colorsystem) of the contained Colors.
		 */
		public function sortByLightness():void	{ palette.sort(sortLightness);	}
		private function sortLightness(c1:Color, c2:Color):Number
		{
			if (c1.lightness < c2.lightness) return -1;
			else if (c1.lightness > c2.lightness) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the value (as set by the HSV Colorsystem) of the contained Colors.
		 */
		public function sortByValue():void	{ palette.sort(sortValue);	}
		private function sortValue(c1:Color, c2:Color):Number
		{
			if (c1.value < c2.value) return -1;
			else if (c1.value > c2.value) return 1;
			return 0;
		}
		
		/**
		 * Sorts the Colors of this Palette by the Hue of the contained Colors
		 */
		public function sortByHue():void	{ palette.sort(sortHue);	}
		private function sortHue(c1:Color, c2:Color):Number
		{
			if (c1.hue < c2.hue) return -1;
			else if (c1.hue > c2.hue) return 1;
			return 0;
		}
		
		private var compareColor:Color;
		/**
		 * Sorts the Colors of this Palette by the proximity of the contained Colors to a certain Color
		 * @param	color
		 */
		public function sortByProximity(color:Color):void	{
			compareColor = color;
			palette.sort(sortProximity);
			compareColor = undefined;
		}
		private function sortProximity(c1:Color, c2:Color):Number
		{
			var proximity1:Number = Math.pow(compareColor.r - c1.r, 2) + Math.pow(compareColor.g - c1.g, 2) + Math.pow(compareColor.b - c1.b, 2);
			var proximity2:Number = Math.pow(compareColor.r - c2.r, 2) + Math.pow(compareColor.g - c2.g, 2) + Math.pow(compareColor.b - c2.b, 2);
			if (proximity1 < proximity2) return -1;
			else if (proximity1 > proximity2) return 1;
			return 0;
		}
		
		/**
		 * Returns a copy of this Palette
		 * @return copy of the original Palette Instance
		 */
		public function clone():Palette
		{
			var clone:Palette = new Palette();
			clone.palette = palette;
			clone.pointer = pointer;
			return clone;
		}
		
	}
	
}