﻿package com.kraftner.geom 
{
	import com.gskinner.utils.Rndm;
	import com.kraftner.utils.NumberUtils;
	
	/**
	 * @internal
	 * Color.as
	 * Copyright (c) 2008 Thomas Kräftner
	 * 
	 * Visit http://blog.kraftner.com for documentation, updates and more free code.
	 */
	
	/**
	* Represents one RGBA Color that can be manipulated in different ways.
	* <p>Instances of this class can be combined to Palettes.</p>
	* <p>This Class also offers a wide variety of manipulating this RGBA color. Keep in mind that some of these manipulations aren't exact and non-reversible which means that setting a certain value +10 and then -10 doesn't necessarily bring you back to the initial value.</p>
	* @see com.kraftner.geom.Palette
	* @see com.kraftner.geom.StaticPalette
	* @author Thomas Kräftner
	*/
	public class Color 
	{
		
		private var _r:Number;
		private var _g:Number;
		private var _b:Number;
		private var _a:Number;
		
		/**
		 * Constructor method
		 * @param	red red value of the RGBA Color
		 * @param	green green value of the RGBA Color
		 * @param	blue blue value of the RGBA Color
		 * @param	alpha alpha value of the RGBA Color
		 */
		public function Color(red:Number=0, green:Number=0, blue:Number=0, alpha:Number=1) 
		{
			r = red;
			g = green;
			b = blue;
			a = alpha;
		}
		
		//////////////////////////////Public Methods//////////////////////////////
		
		/**
		 * Randomly change the red, green, blue and alpha values of this Color.
		 * @param	randomize_red
		 * @param	randomize_green
		 * @param	randomize_blue
		 * @param	randomize_alpha
		 */
		public function random(randomize_red:Boolean = true,randomize_green:Boolean = true,randomize_blue:Boolean = true,randomize_alpha:Boolean=false):void
		{
			if(randomize_red) r = Rndm.integer(0, 255);
			if(randomize_green) g = Rndm.integer(0, 255);
			if(randomize_blue) b = Rndm.integer(0, 255);
			if(randomize_alpha) a = Rndm.float(0, 1);
		}
		
		/**
		 * Returns a copy of this Color
		 * @return copy of the original Color Instance
		 */
		public function clone():Color
		{
			return new Color(r, g, b, a);
		}
		
		/**
		 * Returns the hex code of this Color
		 * @return
		 */
		public function toString():Number
		{
			return hex;
		}
			
		//////////////////////////////Getters and Setters//////////////////////////////
		
		/**
		 * red part of this Color
		 * <p>Only takes values between 0 and 255, Higher/Lower values are trimmed to 0/255.</p>
		 */
		public function get r():Number { return _r; }
		/**
		 * @private
		 */
		public function set r(value:Number):void { _r = NumberUtils.limitValue(value,255); }
		
		/**
		 * green part of this Color
		 * <p>Only takes values between 0 and 255, Higher/Lower values are trimmed to 0/255.</p>
		 */
		public function get g():Number { return _g; }
		/**
		 * @private
		 */
		public function set g(value:Number):void { _g = NumberUtils.limitValue(value,255); }
		
		/**
		 * blue part of this Color
		 * <p>Only takes values between 0 and 255, Higher/Lower values are trimmed to 0/255.</p>
		 */
		public function get b():Number { return _b; }
		/**
		 * @private
		 */
		public function set b(value:Number):void { _b = NumberUtils.limitValue(value,255);
		}
		
		/**
		 * alpha part of this Color
		 * <p>Only takes values between 0 and 1, Higher/Lower values are trimmed to 0/1.</p>
		 */
		public function get a():Number { return _a; }
		/**
		 * @private
		 */
		public function set a(value:Number):void { _a = NumberUtils.limitValue(value,1); }
		
		/**
		 * hex representation of this Color
		 */
		public function get hex():Number { return RGBtoHEX(r, g, b); }
		/**
		 * private
		 */
		public function set hex(value:Number):void
		{
			value = NumberUtils.limitValue(value, 16777215);
			r = value >> 16;
			g = (value >> 8) & 0xff;
			b = value & 0xff;
		}
		
		/**
		 * brightness of this Color
		 * <p>Only takes values between 0 and 100, Higher/Lower values are trimmed to 0/100.</p>
		 */
		public function get brightness():Number { return (r + g + b) / 3 / 2.55; }
		/**
		 * @private
		 */
		public function set brightness(value:Number):void {
			value= NumberUtils.limitValue(value,100)
			var x:Number = Math.min(.001,value/10);
			if (value < brightness) x*= -1;
			while (Math.abs(value - brightness)>=x)
			{
				r += x;
				g += x;
				b += x;
			}
		}
		
		/**
		 * saturation of this Color as set by the HSV Colorsystem
		 * <p>Only takes values between 0 and 100, Higher/Lower values are trimmed to 0/100.</p>
		 */
		public function get saturation_HSV():Number
		{
		  if (max == 0) return 0;
		  else return 100 * (max - min) / max;
		}
		/**
		 * @private
		 */
		public function set saturation_HSV(value:Number):void
		{
			setHSV(hue, NumberUtils.limitValue(value,100),this.value);
		}
		
		/**
		 * saturation of this Color as set by the HSL Colorsystem
		 * <p>Only takes values between 0 and 100, Higher/Lower values are trimmed to 0/100.</p>
		 */
		public function get saturation_HSL():Number
		{
		  var s:Number;
		  var l:Number = lightness / 100;
		  
		  if (max == min) s = 0;
		  else if (l <= .5) s = (max - min) / (2 * l);
		  else if (l > .5) s = (max - min) / (2 - 2 * l);
		 
		  return s*100; 
		}
		/**
		 * @private
		 */
		public function set saturation_HSL(value:Number):void
		{
			setHSL(hue, NumberUtils.limitValue(value,100),lightness);
		}
		
		/**
		 * lightness of this Color as set by the HSL Colorsystem
		 * <p>Only takes values between 0 and 100, Higher/Lower values are trimmed to 0/100.</p>
		 */
		public function get lightness():Number { return (.5 * (max + min) * 100); }
		/**
		 * @private
		 */
		public function set lightness(value:Number):void { setHSL(hue, saturation_HSL, NumberUtils.limitValue(value, 100)); }
		
		/**
		 * value of this Color as set by the HSV Colorsystem
		 * <p>Only takes values between 0 and 100, Higher/Lower values are trimmed to 0/100.</p>
		 */
		public function get value():Number { return max*100; }
		public function set value(value:Number):void { setHSV(hue, saturation_HSV, NumberUtils.limitValue(value, 100)); }
		
		/**
		 * Hue of this Color
		 * <p>Only takes values between 0 and 360, Higher/Lower values are wrapped to be within 0 and 360: 400 would be 40 and -30 would be 330.</p>
		 */
		public function get hue():Number
		{
			var red:Number = r / 255;
			var green:Number = g / 255;
			var blue:Number = b / 255;
			
			var delta:Number = max - min;
				
			var H:Number;
			if (delta==0)
			{
			   H = 0
			}else{
			   if (max == red)
			   {
				   if (g >= b)
				   {
					   H = 60 * ((green - blue) / delta);
				   }else
				   {
					   H = 60 * ((green - blue) / delta) +360;
				   }
			   }else if (max == green)
			   {
				H = 60 * ((blue - red) / delta) +120;
			   }else if (max == blue)
			   {
				H = 60 * ((red - green) / delta) +240;
			   }
			}
			return H;
			
		}
		/**
		 * @private
		 */
		public function set hue(value:Number):void { setHSL(NumberUtils.limitValue(value, 360,true), saturation_HSL,lightness); }
		
		//////////////////////////////Public Helper Methods//////////////////////////////
		
		/**
		 * Converts a RGB color from red, green and blue parts to a HEX representation
		 * @param	red red part of the RGB color
		 * @param	green green part of the RGB color
		 * @param	blue blue part of the RGB color
		 * @return HEX color
		 */
		public static function RGBtoHEX(red:Number, green:Number, blue:Number):Number
		{
			var tempr:String = red.toString(16).toUpperCase();
			if (tempr.length < 2) tempr = "0" + tempr;
			else if (tempr.length > 2) tempr = "FF";
			var tempg:String =green.toString(16).toUpperCase();
			if (tempg.length < 2) tempg = "0" + tempg;
			else if (tempg.length > 2) tempg = "FF";
			var tempb:String = blue.toString(16).toUpperCase();
			if (tempb.length < 2) tempb = "0" + tempb;
			else if (tempb.length > 2) tempb = "FF";
			return Number("0x" + tempr + tempg + tempb);
		}
		
		//////////////////////////////Private Helper Methods//////////////////////////////
		
		private function get max():Number
		{
			var red:Number = r / 255;
			var green:Number = g / 255;
			var blue:Number = b / 255;
			
			return Math.max(red, green, blue);
		}
		
		private function get min():Number
		{
			var red:Number = r / 255;
			var green:Number = g / 255;
			var blue:Number = b / 255;
			
			return Math.min(red, green, blue);
		}
			
		private function setHSV(h:Number, s:Number, v:Number):void
		{
			h = NumberUtils.limitValue(h, 360);
			s = NumberUtils.limitValue(s, 100);
			v = NumberUtils.limitValue(v, 100);
			
			h = h / 360;
			s = s / 100;
			v = v / 100;
			
			if (s == 0)
			{
				r = v * 255;
				g = v * 255;
				b = v * 255;
			}else
			{
				var var_h:Number = h * 6;
				if ( var_h == 6 ) var_h = 0;      //H must be < 1
				var var_i:Number = int( var_h );             //Or ... var_i = floor( var_h )
				var var_1:Number = v * ( 1 - s );
				var var_2:Number = v * ( 1 - s * ( var_h - var_i ) );
				var var_3:Number = v * ( 1 - s * ( 1 - ( var_h - var_i ) ) );
				var var_r:Number;
				var var_g:Number;
				var var_b:Number;
				if ( var_i == 0 ){
					var_r = v;
					var_g = var_3;
					var_b = var_1
				}
				else if ( var_i == 1 ) 
				{
					var_r = var_2 ;
					var_g = v;
					var_b = var_1;
				}
				else if ( var_i == 2 )
				{
					var_r = var_1;
					var_g = v;
					var_b = var_3;
				}
				else if ( var_i == 3 )
				{
					var_r = var_1;
					var_g = var_2;
					var_b = v;
				}
				else if ( var_i == 4 )
				{
					var_r = var_3;
					var_g = var_1;
					var_b = v
				}
				else 
				{
					var_r = v;
					var_g = var_1;
					var_b = var_2;
				}
				r = var_r * 255;                  //RGB results from 0 to 255
				g = var_g * 255;
				b = var_b * 255;
				
			}
			
		}
		
		private function setHSL(h:Number, s:Number,l:Number):void
		{
			h = NumberUtils.limitValue(h, 360);
			s = NumberUtils.limitValue(s, 100);
			l = NumberUtils.limitValue(l, 100);
			
			h = h / 360;
			s = s / 100;
			l = l / 100;
				
			if (s == 0)
			{
				r = l * 255;
				g = l * 255;
				b = l * 255;
			}else
			{
				var var_1:Number;
				var var_2:Number;
				if ( l < 0.5 )
				{
					var_2 = l * ( 1 + s );
				}else
				{
					var_2 = ( l + s ) - ( s * l );
				}
			   var_1 = 2 * l - var_2;
			   r = 255 * HUEtoRGB( var_1, var_2, h + ( 1 / 3 ) );
			   g = 255 * HUEtoRGB( var_1, var_2, h );
			   b = 255 * HUEtoRGB( var_1, var_2, h - ( 1 / 3 ) );
				
			}
		}
		
		private function HUEtoRGB(v1:Number, v2:Number, vH:Number):Number
		{
			if ( vH < 0 ) vH += 1;
			if ( vH > 1 ) vH -= 1;
			if ( ( 6 * vH ) < 1 ) return ( v1 + ( v2 - v1 ) * 6 * vH );
			if ( ( 2 * vH ) < 1 ) return ( v2 );
			if ( ( 3 * vH ) < 2 ) return ( v1 + ( v2 - v1 ) * ( ( 2 / 3 ) - vH ) * 6 );
			return ( v1 )
		}
		
	}
	
}